/*:
 * @target MZ
 * @plugindesc v1.3 言語別フォント切替（HS_LangSimple連動）／サイズは保持・任意で縁取りとサイズ倍率も調整
 * @author HS
 *
 * @help
 * - HS_LangSimple の ConfigManager.hs_langIndex（0:JP 1:EN 2:zh-CN 3:ko-KR）
 *   に連動してフォントファミリを切り替えます。
 * - 既存ウィンドウの fontSize 等は維持し、fontFace だけ差し替えます。
 * - 一部プラグインが $gameSystem.mainFontFace() を参照する対策も含みます。
 * - gamefont.css で下記 family を定義しておいてください（例）：
 *     GameFontJP / GameFontSC / GameFontKR
 *
 * 【設置順】HS_LangSimple の「下」。メッセージ系プラグインよりも下推奨。
 *
 * @param FamilyJP
 * @text JPファミリ
 * @type string
 * @default GameFontJP
 *
 * @param FamilyEN
 * @text ENファミリ
 * @type string
 * @default GameFont
 *
 * @param FamilySC
 * @text ZH-CNファミリ
 * @type string
 * @default GameFontSC
 *
 * @param FamilyKR
 * @text KOファミリ
 * @type string
 * @default GameFontKR
 *
 * @param TouchOutline
 * @text 縁取りも適用
 * @type boolean
 * @default false
 * @desc ONなら、下の各言語の縁取り設定（幅/色）も適用します。
 *
 * @param OutlineJP
 * @text JP縁取り
 * @type struct<Outline>
 * @default {"w":"4","c":"rgba(0,0,0,0.6)"}
 *
 * @param OutlineEN
 * @text EN縁取り
 * @type struct<Outline>
 * @default {"w":"4","c":"rgba(0,0,0,0.6)"}
 *
 * @param OutlineSC
 * @text ZH-CN縁取り
 * @type struct<Outline>
 * @default {"w":"4","c":"rgba(0,0,0,0.6)"}
 *
 * @param OutlineKR
 * @text KO縁取り
 * @type struct<Outline>
 * @default {"w":"4","c":"rgba(0,0,0,0.6)"}
 *
 * @param UseSizeScale
 * @text 言語別サイズ倍率を使う
 * @type boolean
 * @default false
 *
 * @param ScaleJP
 * @parent UseSizeScale
 * @text JPサイズ倍率
 * @type number
 * @decimals 2
 * @default 1.00
 *
 * @param ScaleEN
 * @parent UseSizeScale
 * @text ENサイズ倍率
 * @type number
 * @decimals 2
 * @default 1.00
 *
 * @param ScaleSC
 * @parent UseSizeScale
 * @text ZHサイズ倍率
 * @type number
 * @decimals 2
 * @default 1.00
 *
 * @param ScaleKR
 * @parent UseSizeScale
 * @text KRサイズ倍率
 * @type number
 * @decimals 2
 * @default 1.00
 */
/*~struct~Outline:
 * @param w
 * @text 幅
 * @type number
 * @default 4
 * @param c
 * @text 色
 * @type string
 * @default rgba(0,0,0,0.6)
 */
(() => {
  "use strict";
  const PN = "HS_FontByLang";
  const P  = PluginManager.parameters(PN);

  const FAMILY = {
    0: String(P.FamilyJP || "GameFontJP"),
    1: String(P.FamilyEN || "GameFont"),
    2: String(P.FamilySC || "GameFontSC"),
    3: String(P.FamilyKR || "GameFontKR"),
  };

  const TOUCH_OUTLINE = String(P.TouchOutline || "false") === "true";
  function parseOutline(s) {
    try { return JSON.parse(s); } catch(_) { return {w:4, c:"rgba(0,0,0,0.6)"}; }
  }
  const OUTLINE = {
    0: parseOutline(P.OutlineJP || '{"w":"4","c":"rgba(0,0,0,0.6)"}'),
    1: parseOutline(P.OutlineEN || '{"w":"4","c":"rgba(0,0,0,0.6)"}'),
    2: parseOutline(P.OutlineSC || '{"w":"4","c":"rgba(0,0,0,0.6)"}'),
    3: parseOutline(P.OutlineKR || '{"w":"4","c":"rgba(0,0,0,0.6)"}'),
  };

  const USE_SIZE_SCALE = String(P.UseSizeScale || "false") === "true";
  const SCALE = {
    0: Number(P.ScaleJP || 1.00),
    1: Number(P.ScaleEN || 1.00),
    2: Number(P.ScaleSC || 1.00),
    3: Number(P.ScaleKR || 1.00),
  };

  // 現在の言語index（HS_LangSimple が ConfigManager.hs_langIndex に保持）
  function langIndex() {
    return Number(ConfigManager && ConfigManager.hs_langIndex != null
      ? ConfigManager.hs_langIndex : 0);
  }

  const familyNow  = () => FAMILY[langIndex()] || FAMILY[0] || "GameFont";
  const outlineNow = () => OUTLINE[langIndex()] || OUTLINE[0] || {w:4,c:"rgba(0,0,0,0.6)"};
  const sizeScaleNow = () => Number(SCALE[langIndex()] || 1.00);

  // 一部プラグインの mainFontFace() 対策：ここも言語別に返す
  const _mainFontFace = Window_Base.prototype.standardFontFace || Window_Base.prototype.standardFontFace;
  Window_Base.prototype.standardFontFace = function() {
    // MZのデフォは FontManager.face
    return familyNow();
  };

  // サイズ倍率がONなら standardFontSize を倍率適用
  if (USE_SIZE_SCALE) {
    const _standardFontSize = Window_Base.prototype.standardFontSize;
    Window_Base.prototype.standardFontSize = function() {
      const base = _standardFontSize.call(this);
      return Math.round(base * sizeScaleNow());
    };
  }

  // ここが肝：サイズ等を壊さず fontFace だけ差し替える
  function applyFontChange(forceOutline = false) {
    try {
      FontManager.clear();
      // できれば事前読み込み（存在チェックとしても有効）
      if (document.fonts) {
        ["GameFontJP","GameFontSC","GameFontKR", familyNow()]
          .filter((v,i,a)=>v && a.indexOf(v)===i)
          .forEach(f=>{ try { document.fonts.load(`16px "${f}"`); } catch(_){} });
      }

      const fam = familyNow();
      const scene = SceneManager._scene;
      if (!scene) return;

      const walk = (node) => {
        if (!node) return;
        if (node instanceof Window_Base) {
          const bmp = node.contents;
          if (bmp) {
            // 既存スタイルを保持
            const keep = {
              size: bmp.fontSize,
              color: bmp.textColor,
              oWidth: bmp.outlineWidth,
              oColor: bmp.outlineColor
            };
            // resetFontSettings() は呼ばない → サイズ初期化を防ぐ
            bmp.fontFace = fam;
            bmp.fontSize = keep.size;
            bmp.textColor = keep.color;

            if (TOUCH_OUTLINE || forceOutline) {
              const ol = outlineNow();
              bmp.outlineWidth = Number(ol.w ?? keep.oWidth);
              bmp.outlineColor = String(ol.c ?? keep.oColor);
            } else {
              bmp.outlineWidth = keep.oWidth;
              bmp.outlineColor = keep.oColor;
            }
          }
          if (typeof node.refresh === "function") {
            try { node.refresh(); } catch(_) {}
          }
        }
        if (Array.isArray(node.children)) node.children.forEach(walk);
      };
      walk(scene);
    } catch (e) {
      console.warn("HS_FontByLang: applyFontChange error", e);
    }
  }

  // 言語変更の検知：Config save と毎フレーム
  let _last = langIndex();

  const _configSave = ConfigManager.save;
  ConfigManager.save = function() {
    const res = _configSave.call(this);
    const now = langIndex();
    if (now !== _last) { _last = now; applyFontChange(); }
    return res;
  };

  const _updateMain = SceneManager.updateMain;
  SceneManager.updateMain = function() {
    _updateMain.call(this);
    const now = langIndex();
    if (now !== _last) { _last = now; applyFontChange(); }
  };

  // 起動時
  const _bootStart = Scene_Boot.prototype.start;
  Scene_Boot.prototype.start = function() {
    _bootStart.call(this);
    applyFontChange();
  };

  // --- 初回タイトルの取りこぼし防止（即時 + 次フレームで再適用） ---
  const _titleCreate = Scene_Title.prototype.create;
  Scene_Title.prototype.create = function() {
    _titleCreate.call(this);
    try { applyFontChange(true); } catch(_) {}
    setTimeout(() => { try { applyFontChange(true); } catch(_) {} }, 0);
  };

  const _titleStart = Scene_Title.prototype.start;
  Scene_Title.prototype.start = function() {
    _titleStart.call(this);
    try { requestAnimationFrame(() => applyFontChange(true)); } catch(_) {}
  };

})();
